module RevWiki

	class PageProtoType
		attr_reader :number    # ファイル番号
		attr_accessor :editor  # （最新の）編集者
		attr_accessor :summary # 更新の概要
		attr_accessor :rev     # メジャーリビジョン
		attr_accessor :fix     # マイナーリビジョン
		attr_accessor :update  # （最新の）更新日時

		include RevWiki::Regulation

		NOTE_PREFIX = "ノート:"
		MYPAGE_PREFIX = "ユーザー:"
		MYPAGE_PREFIX_PATTERN = /^#{Regexp.escape(MYPAGE_PREFIX)}(.+)/
		SPECIAL_PREFIX = "特殊:"
		SPECIAL_PREFIX_PATTERN = /^#{Regexp.escape(SPECIAL_PREFIX)}(.+)/
		KEYWORD_PREFIX = "キーワード:"
		KEYWORD_PREFIX_PATTERN = /^#{Regexp.escape(KEYWORD_PREFIX)}(.+)/
		EDIT_PREFIX = "編集:"
		EDIT_PREFIX_PATTERN = /^#{Regexp.escape(EDIT_PREFIX)}(.+)/
		SPECIAL_NAMES = {}
		SPECIAL_NAMES[:format_help] = "#{SPECIAL_PREFIX}整形ルール"
		SPECIAL_NAMES[:keyword] = "#{SPECIAL_PREFIX}キーワード"
		SPECIAL_NAMES[:plugin] = "#{SPECIAL_PREFIX}プラグイン"


		# リビジョンを文字列で返す
		def rev_full
			return sprintf("%d.%d", @rev, @fix)
		end

		# そのページが存在するページかどうかを調べる
		# （No.0なら存在しないページ）
		def exist?
			return @number != 0
		end


	end


	class Page < PageProtoType
		attr_accessor :name         # ページ名
		attr_accessor :score        # スコア
		attr_accessor :major_update # （最新の）主要な更新日時
		attr_accessor :frozen       # 凍結フラグ

		def initialize(number = 0, name = "")
			@number = number
			@name = name
			@editor = ""
			@summary = ""
			@rev = 1
			@fix = 0
			@update = nil
			@major_update = nil
			@score = 100
			@frozen = false
			@revisions = nil
		end

		# 本文ファイル名
		def content_file_name
			return RevWiki.number_to_file_name(@number, 'wiki')
		end

		# 本文ファイルパス（Pathnameオブジェクト）
		def content_file_path
			return (DIR_PATHS[:data] + self.content_file_name)
		end

		# タグファイル名
		def tag_file_name
			return number_to_file_name(@number, 'tag')
		end

		# タグファイルパス（Pathnameオブジェクト）
		def tag_file_path
			return (DIR_PATHS[:data] + self.tag_file_name)
		end


		# リビジョンや本文まで含めた総ファイルサイズ
		def total_file_size
			file_paths = DIR_PATHS[:data].children.select{|p| p.to_s =~ Regexp.new(number_to_file_name(@number, '*'))}
			total_size = 0

			file_paths.each do |path|
				total_size += path.size
			end
			return total_size
		end

		# 閲覧されたときに呼ぶべきメソッド
		def on_viewed
			RevWiki.operate_store(Page.record_file_path){|record|
				last_viewed = (record['last_viewed'] || Time.now)

				record['today_view_counts'] ||= {}
				record['yesterday_view_counts'] ||= {}
				record['today_view_hosts'] ||= {}
				record['yesterday_view_hosts'] ||= {}
				record['today_referers'] ||= {}
				record['yesterday_referers'] ||= {}
	
				if last_viewed.strftime("%x") != Time.now.strftime("%x") then
					# 日付が変わっている場合の処理
					record['yesterday_view_counts'] = record['today_view_counts'].dup
					record['today_view_counts'] = {}
					record['yesterday_view_hosts'] = record['today_view_hosts'].dup
					record['today_view_hosts'] = {}
					record['yesterday_referers'] = record['today_referers'].dup
					record['today_referers'] = {}
				end

				# 今日のカウント
				record['today_view_counts'][@number] ||= 0
				record['today_view_counts'][@number] += 1
				if $cgi.referer && ($cgi.referer != "") &&
				!($cgi.referer =~ /#{URLS[:base]}/) then
					record['today_referers'][@number] ||= {}
					record['today_referers'][@number][$cgi.referer] ||= 0
					record['today_referers'][@number][$cgi.referer] += 1
				end
				record['today_view_hosts'][@number] ||= []
				unless record['today_view_hosts'][@number].include?($cgi.remote_addr) then
					record['today_view_hosts'][@number] << $cgi.remote_addr
				end
				
				record['last_viewed'] = Time.now
			}

		end

		def today_view_count
			counts = ( RevWiki.load_store(Page.record_file_path, 'today_view_counts') || {} )
			return( counts[@number] || 0 )
		end

		def yesterday_view_count
			counts = ( RevWiki.load_store(Page.record_file_path, 'yesterday_view_counts') || {} )
			return( counts[@number] || 0 )
		end

=begin
		# スコア加算
		def incliment_score(value)
			LOG.puts "Score Incliment #{@name}"
			LOG.indent_level += 1
			LOG.puts "Base Incliment: #{value}"

			# 更新頻度修正：
			#   本来の増分 × （1.00 ＋ 7日以内の更新回数×0.05 ＋ 8日～３０日以内の更新回数×0.02）
			rate = 1.00
			revisions = Revision.get_revisions(self)
			revisions.each do |revision|
				sec = self.update - revision.update
				if sec <= (60 * 60 * 24 * 7).to_f then
					rate += 0.05
				elsif sec <= (60 * 60 * 24 * 30).to_f then
					rate += 0.02
				end
			end
			value *= rate
			LOG.puts "Updation Bonus: ×#{rate}"

			
			self.score += value
			LOG.puts "Total Increment: #{value}"

			# 平均化修正
			total_score = 0.00
			pages = []
			PAGE_NUMBER_TABLE.each_pair do |name, number|
				pages << Page.load_tag(number)
				total_score += pages.last.score
			end
			pages.each do |page|
				page.score *= ((pages.size * 100) / total_score)
				LOG.puts "#{page.name} = #{page.score}pts"
				page.save_tag
			end

			LOG.indent_level -= 1
		end
=end

		# リビジョン配列（1.0から順に並んでいる）
		def revisions
			@revisions = Revision.get_revisions(@number) unless @revisions

			return @revisions
		end


		# 本文をファイルから読み出す
		# 返り値はUTF-8であるべき
		def load_content(log_message = nil)
			if self.content_file_path.exist? then
				return RevWiki.file_lock{ open(content_file_path, 'r'){|f| UTF8String.new(f.read)} }
			else
				return ""
			end
		end

		# 本文をファイルに書き込む
		def save_content(content, log_message = nil)
			file_lock{
				open(content_file_path, 'w'){|file|
					file.write(content)
				}
			}
		end

		# タグファイルを書き込む
		def save_tag(log_message = nil)
			operate_store(self.tag_file_path){|db|
				db['name'] = @name
				db['editor'] = @editor
				db['summary'] = @summary
				db['rev'] = @rev
				db['fix'] = @fix
				db['update'] = @update
				db['major_update'] = @major_update
				db['score'] = @score
				db['frozen'] = @frozen
			}
		end




		# データディレクトリの中の*.tagファイルを全て読み込み
		# ページテーブル（Hash）を構築する
		def Page.get_page_table
			pattern = /^#{FILE_PREFIX}_([0-9]+)\.tag$/ # rw_00001.tag
			table = {}
			
			DIR_PATHS[:data].children.each do |path|
				if path.basename.to_s =~ pattern then
					page = Page.load_tag($1.to_i)

					# 重複チェック
					raise(RevWikiError, "page-names overlapping（ページ名が重複しています） : #{page.name}") if table.include?(page.name)

					table[page.name] = page
				end
			end

			return table
		end

		# 番号テーブルを構築する
		def Page.get_number_table
			pattern = /^#{FILE_PREFIX}_([0-9]+)\.tag$/ # rw_00001.tag
			table = {}
			
			DIR_PATHS[:data].children.each do |path|
				if path.basename.to_s =~ pattern then
					name = RevWiki.load_store(path, 'name')

					# 重複チェック
					raise(RevWikiError, "page-names overlapping（ページ名が重複しています） : #{name}") if table.include?(name)

					table[name] = $1.to_i
				end
			end

			return table

		end


		# タグを読み出してPageオブジェクトを作成
		def Page.load_tag(number, log_message = nil)
			path = DIR_PATHS[:data] + RevWiki.number_to_file_name(number, 'tag')
			page = Page.new(number)
			RevWiki.operate_store(path){|db|
				page.name = db['name']
				page.editor = db['editor']
				page.summary = db['summary']
				page.rev = db['rev']
				page.fix = db['fix']
				page.major_update = (db['major_update'] || db['update'])
				page.update = db['update']
				page.score = db['score']
				page.frozen = db['frozen']
			}

			return page
		end

		# ページ名を正規化
		def Page.filt_name(page_name)
			return "" unless page_name
			page_name = page_name.dup
			page_name.gsub!(/：/, ":")
			page_name.gsub!(/＠/, "@")
			page_name.gsub!(/／/, "/")
			page_name.gsub!(/[ 　]/, "_")
			return page_name
		end

		# ページ名のアンダーバーを半角スペースに変換
		def Page.filt_name_for_display(page_name)
			page_name = page_name.dup
			page_name.gsub!(/_/, " ")
			return page_name
		end


		# アクセスレコードのファイル名
		def Page.record_file_name
			return "access_record.dat"
		end

		# アクセスレコードのファイルパス
		def Page.record_file_path
			return Pathname.new(DIR_PATHS[:data] + Page.record_file_name)
		end

	end

	class SpecialPage < Page
		def revisions
			return []
		end
	end


	class Revision < PageProtoType
		attr_accessor :patch

		def initialize(number = 0, page = nil)
			@number = number
			reflect_page(page) if page
		end

		def reflect_page(page)
			@rev = page.rev
			@fix = page.fix
			@editor = page.editor
			@summary = page.summary
			@update = page.update
		end

		# ファイル名
		def file_name
			return number_to_file_name(@number, "patch", @rev)
		end

		# ファイルパス（Pathnameオブジェクト）
		def file_path
			return (DIR_PATHS[:data] + self.file_name)
		end

		def revisions
		end

		# リビジョンファイルを書き出す
		def save(patch, log_message = nil)
			data = {}
			data['editor'] = @editor
			data['summary'] = @summary
			data['update'] = @update
			data['patch'] = patch

			operate_store(self.file_path){|db|
				db['revisions'] = [] unless db['revisions']
				db['revisions'][@fix] = data
			}

		end

		# 本文を復元する
		def load_content(message = nil)
			revisions = Page.load_tag(@number).revisions
			
			# 自分より前のリビジョンは全て除外
			revisions.delete_if do |revision|
				@rev > revision.rev || (@rev == revision.rev && @fix > revision.fix)
			end


			revisions.reverse!
			content_lines = Page.load_tag(@number).load_content.gsub(/\r\n/, "\n").split("\n")

			revisions.each do |revision|
				content_lines = Diff::LCS.unpatch!(content_lines, revision.patch)
			end

			return content_lines.join("\n")
		end

		# リビジョンファイルを読み込む
		# 指定したリビジョンが存在しなければnilが返る
		def Revision.load(number, rev, fix, log_message = nil)
			path = DIR_PATHS[:data] + RevWiki.number_to_file_name(number, 'patch', rev)
			return nil unless path.exist?
			

			revision = Revision.new(number)

			begin
				RevWiki.operate_store(path){|db|
					revision.editor = db['revisions'][fix]['editor']
					revision.summary = db['revisions'][fix]['summary']
					revision.update = db['revisions'][fix]['update']
					revision.patch = db['revisions'][fix]['patch']
				}
			rescue
				return nil
			end

			revision.rev = rev
			revision.fix = fix


			return revision
		end



		# リビジョンの数値を一つ落とす。返り値はハッシュ
		# ファイルが見つからなかった場合、エラーマークとしてrev0.0を返す
		def Revision.decliment_rev(page, rev, fix)
			# 引数がページ番号のときの処理
			page = Page.load_tag(page) if page.is_a?(Integer)

			if rev == 1 && fix == 0 then
				return nil
			elsif fix > 0 then
				return rev, (fix - 1)
			else
				path = DIR_PATHS[:data] + RevWiki.number_to_file_name(page.number, "patch", rev - 1)
				revisions = RevWiki.load_store(path, 'revisions')

				if revisions then
					return (rev - 1), (revisions.size - 1)
				else
					return 0, 0
				end
			end
		end


		# 指定したページの全リビジョンを配列で返す（1.0～）
		def Revision.get_revisions(page, major_only = false)

			# 引数がページ番号のときの処理
			page = Page.load_tag(page) if page.is_a?(Integer)

			revisions = []
			rev = page.rev
			fix = page.fix


			# メインループ
			until rev == 1 && fix == 0 do
				rev, fix = Revision.decliment_rev(page, rev, fix)
				break if rev == 0 # リビジョンファイルが存在しなければ終了

				revision = Revision.load(page.number, rev, fix)
				revisions.unshift(revision) if revision

				fix = 0 if major_only
			end


			return revisions
		end

	end
end
