# Userクラス・EditHistoryクラスの定義（レベル２）
module RevWiki
	class UserProtoType
		attr_accessor :prefs
		include Regulation
	end

	class EntryUser < UserProtoType
		attr_reader :number
		attr_accessor :name, :edit_history

		def initialize(number, password = "")
			@number = number
			@name = ""
			@password_crypt = password.crypt(SALT)
			@edit_history = []
			@prefs = DEFAULT_PREFS.dup
		end

		# ファイル名
		def file_name
			return sprintf("user_%04d.tag", @number)
		end

		# ファイルパス（Pathnameオブジェクト）
		def file_path
			return (DIR_PATHS[:data] + self.file_name)
		end

		# 管理権限があるかどうか
		def admin?
			return(self.entry? && ADMINISTRATORS.include?(@name))
		end

		# 登録ユーザーかどうか
		def entry?
			return true
		end

		def password_match?(password)
			return password.crypt(SALT) == @password_crypt
		end

		def save(log_message = nil)
			operate_store(self.file_path){|db|
				db['name'] = @name
				db['password_crypt'] = @password_crypt
				db['edit_history'] = @edit_history
				db['summary'] = @summary
				db['prefs'] = @prefs
			}
		end

		def load
			RevWiki.operate_store(self.file_path){|db|
				@name = db['name']
				@password_crypt = db['password_crypt']
				@edit_history = db['edit_history']
				@summary = db['summary']
				@prefs = db['prefs']
			}
			return self
		end

		def EntryUser.load(number, log_message = nil)
			if number == 0 then
				return AnonymousUser.new
			end
			path = DIR_PATHS[:data] + sprintf("user_%04d.tag", number)
			user = EntryUser.new(number).load
			return user
		end


		def EntryUser.get_user_number_table
			pattern = /^user_([0-9]+)\.tag$/ # user_0001.tag
			table = {}
			
			DIR_PATHS[:data].children.each do |path|
				if path.basename.to_s =~ pattern then
					name = RevWiki.load_store(path, 'name')

					# 重複チェック
					raise(RevWikiError, "user-names overlapping（ユーザー名が重複しています） : #{name}") if table.include?(name)

					table[name] = $1.to_i
				end
			end

			return table
		end

	end

	class AnonymousUser < UserProtoType
		def initialize
			@prefs = ($session['prefs'] || DEFAULT_PREFS.dup)
		end

		def number
			return 0
		end

		def name
			return ($cgi.remote_addr || "anonymous")
		end

		def file_name
			return ""
		end

		def file_path
			return ""
		end

		def entry?
			return false
		end

		def admin?
			return false
		end

		def save
			$session['prefs'] = @prefs
			return true
		end

		def edit_history
			return []
		end
	end

	class EditHistoryItem
		attr_accessor :page_name, :rev, :fix, :update, :summary
		def initialize(page)
			@page_name = page.name
			@rev = page.rev
			@fix = page.fix
			@update = page.update
			@summary = page.summary
		end
	end
end