module RevWiki

	EDITABLE_ONLY = 1     # 編集可能な時のみ選択可能
	UNEDITABLE_ONLY = 2   # 編集不可能な時のみ選択可能
	ANONYMOUS_ONLY = 3    # 非登録ユーザーのみ選択可能
	USER_ONLY = 4         # 登録ユーザーか管理者のみ選択可能
	ADMIN_ONLY = 5        # 管理者のみ選択可能

	VIEW_MODE_ONLY = 6   # 閲覧モード時のみ選択可能
	EDIT_MODE_ONLY = 7   # 編集モード時のみ選択可能

	EXISTING_PAGE_ONLY = 8   # そのページが実際に存在しているときのみ選択可能
	                         # （＝特殊ページ以外では有効）


	class MenuItem
		attr_accessor :caption, :tip, :with_page, :options, :access_key
		attr_writer :path

		def initialize(path, caption, tip, with_page = false, *options)
			@path = Pathname.new(path)
			@caption = caption
			@tip = tip
			@with_page = with_page
			@options = options.flatten
			@access_key = nil
		end
		
		def ==(other)
			if other.is_a?(String)
				return @caption == other
			else
				super(other)
			end
		end
		
		def path
			if @with_page then
				return Pathname.new(sprintf("%s%s", @path, ($current_page ? CGI.escape($current_page.name) : "")))
			else
				return @path
			end
		end
		
		def display?
		  return false if RevWiki.editable? && @options.include?(UNEDITABLE_ONLY)
      return false if !(RevWiki.editable?) && @options.include?(EDITABLE_ONLY)
			return false if @options.include?(ANONYMOUS_ONLY) && $current_user.entry?
			return false if @options.include?(USER_ONLY) && !($current_user.entry?)
			return false if @options.include?(ADMIN_ONLY) && !($current_user.admin?)
			return false if @options.include?(VIEW_MODE_ONLY) && $edit_mode
			return false if @options.include?(EDIT_MODE_ONLY) && !($edit_mode)
			return false if @options.include?(EXISTING_PAGE_ONLY) && $current_page.is_a?(SpecialPage)

			return true
		end



		def to_model(caption_id)

			return nil unless self.display?

			model = {}
			model[:href] = self.path
			model[:title] = @tip + Interface.access_key_hint(@access_key, true)
			model[:accesskey] = @access_key
			model[caption_id.to_sym] = @caption
			model[caption_id.to_sym] += Interface.access_key_hint(@access_key) if $current_user.prefs['visible_access_key']
			return model
		end


	end


	WIKI_MENU_ITEMS = []
	USER_MENU_ITEMS = []
	PAGE_MENU_ITEMS = []


	WIKI_MENU_ITEMS <<
		MenuItem.new("view.cgi", MENU_WORDS[:front_page], MENU_TIPS[:front_page])
	WIKI_MENU_ITEMS.last.access_key = "f"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=create_page", MENU_WORDS[:create_page], MENU_TIPS[:create_page], false, EDIT_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "n"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=list", MENU_WORDS[:list], MENU_TIPS[:list])
	WIKI_MENU_ITEMS.last.access_key = "l"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=pref", MENU_WORDS[:pref], MENU_TIPS[:pref], false, EDIT_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "p"

	WIKI_MENU_ITEMS <<
		MenuItem.new("view.cgi?edit_mode=0&page=", MENU_WORDS[:view_mode], MENU_TIPS[:view_mode], true, EDIT_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "m"

	WIKI_MENU_ITEMS <<
		MenuItem.new("view.cgi?edit_mode=1&page=", MENU_WORDS[:edit_mode], MENU_TIPS[:edit_mode], true, VIEW_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "m"


	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=login", MENU_WORDS[:login], MENU_TIPS[:login], false, ANONYMOUS_ONLY, EDIT_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "u"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=edit_history", MENU_WORDS[:edit_history], MENU_TIPS[:edit_history], false, USER_ONLY, EDIT_MODE_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "h"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=logout", MENU_WORDS[:logout], MENU_TIPS[:logout], false, USER_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "u"

	WIKI_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=admin", MENU_WORDS[:admin_menu], MENU_TIPS[:admin_menu], false, ADMIN_ONLY)
	WIKI_MENU_ITEMS.last.access_key = "a"




	PAGE_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=edit&page=", MENU_WORDS[:edit], MENU_TIPS[:edit], true, EDITABLE_ONLY, EDIT_MODE_ONLY)
	PAGE_MENU_ITEMS.last.access_key = "e"

	PAGE_MENU_ITEMS <<
		MenuItem.new("command.cgi?cmd=revision&page=", MENU_WORDS[:revision], MENU_TIPS[:revision], true, EXISTING_PAGE_ONLY, EDIT_MODE_ONLY)
	PAGE_MENU_ITEMS.last.access_key = "r"


	PAGE_MENU_ITEMS << MenuItem.new("command.cgi?cmd=rename&page=", MENU_WORDS[:rename], MENU_TIPS[:rename], true,
		                              EDITABLE_ONLY, EDIT_MODE_ONLY, USER_ONLY)
	PAGE_MENU_ITEMS << MenuItem.new("command.cgi?cmd=delete&page=", MENU_WORDS[:delete], MENU_TIPS[:delete], true,
		                              EDITABLE_ONLY, EDIT_MODE_ONLY, ADMIN_ONLY)

end

