FORMAT_NAME = "AYANE/1.0"
FORMAT_ID = "ayane10"


class WikiCloth

	attr_reader :text

	def initialize(text)
		@text = text.gsub(/\r\n/, "\n")
	end

	# 行頭マーク
	HEADMARK_BLANK = /^[ \t]*$/
	
	HEADMARK_H2 = /^\!(?!\!\!)[ ]*(.*)/                # ! $+
	HEADMARK_H3 = /^\!\!(?!\!)[ ]*(.*)/                # !! $+
	HEADMARK_H4 = /^\!\!\![ ]*(.*)/                    # !!! $+
	HEADMARK_H2_LINE = /^\=\=\=*\s*$/               # ========
	HEADMARK_H3_LINE = /^\-\-\-*\s*$/               # --------
	
	HEADMARK_PRE = /^\|(.*)|^[ ][ ][ ][ ](.*)/                        # |
	HEADMARK_BLOCKQUOTE = /^\>\ *(.*)/              # > $+
	
	HEADMARK_UL1 = /^\*(?!\*)(?!\*)\ *(.*)/         # *
	HEADMARK_UL2 = /^(?:\*\*(?!\*)|[ ]{4}\*)[ ]*(.*)/
	                                                # ** or ____*
	HEADMARK_UL3 = /^(?:\*\*\*|[ ]{8}\*)[ ]*(.*)/
	                                                # *** or ________*
	HEADMARK_OL1 = /^\+(?!\+)(?!\+)[ ]*(.*)|^[0-9]+\.[ ]*(.*)/
	                                                # + or 1.
	HEADMARK_OL2 = /^(?:\+\+(?!\+)|[ ]{4}\+|[ ]{4}[0-9]+\.)[ ]*(.*)/
	                                                # ++ or 1.
	HEADMARK_OL3 = /^(?:\+\+\+|[ ]{8}\+|[ ]{8}[0-9]+\.)[ ]*(.*)/
	                                                # +++ or 1.
	HEADMARK_DL = /^(.*)\:\:[ ]*$/                    # $+:


	HEADMARK_COMMENT = /^\#/                        # #
	
	HEADMARK_BLOCK_BEGIN = /^\=\=+(.*)/           # ==$+
	HEADMARK_BLOCK_END = /^\=\=+[ ]*$/                 # ==
	
	HEADMARK_PLUGIN = /^\{\{(.*?)\}\}$/             # {{$+}} or #
	
	MARK_BR = /\/\/[ ]*$|\&lt\;br\&gt\;[ ]*$/       # // or <br>
	MARK_EM = /[ ]*[*_]{2}(.*?)[*_]{2}[ ]*/         # **$+** or __$+__
	MARK_STRONG = /[ ]*[*_]{3}(.*?)[*_]{3}[ ]*/     # ***$+*** or ___$+___
	MARK_LINK = /\[\[[ ]*(.*?)[ ]*\]\]/             # [[$+]]
	MARK_LINK_PAREN = /\[\[[ ]*([^\[]*?)[ ]*\]\]\([ ]*(.*?)[ ]*\)/
                                                  # [[$1]]($2)
	MARK_URLLINK = /\[[ ]*(.*?)[ ]*\]/              # [$1]
	MARK_URLLINK_PAREN = /\[[ ]*([^\[]*?)[ ]*\]\([ ]*(.*?)[ ]*\)/
	                                                # [$1]($2)

	MARK_CITE = /[ ]*\'\'(.+?)\'\'[ ]*/             # ''$1''

	MARK_PLUGIN = /\{\{(.*?)\}\}/

	MARK_ID = /\(\([ ]*([A-Za-z][A-Za-z0-9\_\:\.\-]*)[ ]*\)\)/
	                                                # (( $1 ))
	
	MAX_LINE = 1000
	
	
	@section = nil
		
	
	
	def content_to_html
		str = @text
		lines = str.split(/\n/)
		root = HTMLScanner::Root.new(nil, nil)
		i = 0
		buf = ""
		
		MAX_LINE.times do
			line = lines[i]
			ahead_line = lines[i+1]
			
			break unless line
			
			# パラグラフ（段落）------------------------------------
			if @section == :section_p then
				if line =~ HEADMARK_BLANK then
					@section = nil
					i = i + 1; next
				end
			end
			
			# ベタ書き（PRE）---------------------------------------
			if @section == :section_pre then
				if line =~ HEADMARK_PRE then
					HTMLScanner::Text.new(root.children.last, nil, "\n" + CGI.escapeHTML($+))
					i = i + 1; next
				else
					@section = nil
				end
			end
			
			# 引用文（BLOCKQUOTE）----------------------------------
			if @section == :section_blockquote then
				if line =~ HEADMARK_BLOCKQUOTE then
					_add_inline_element(root.children.last, $+)
					i = i + 1; next
				else
					@section = nil
				end
			end
			
			# 順序無しリスト ---------------------------------------
			case @section
			when :section_ul1
				_seq_ul(root.children.last, 1, line)
				i = i + 1; next
			when :section_ul2
				_seq_ul(root.children.last, 2, line)
				i = i + 1; next
			when :section_ul3
				_seq_ul(root.children.last, 3, line)
				i = i + 1; next
			end
			
			# 順序付けリスト ---------------------------------------
			case @section
			when :section_ol1
				_seq_ol(root.children.last, 1, line)
				i = i + 1; next
			when :section_ol2
				_seq_ol(root.children.last, 2, line)
				i = i + 1; next
			when :section_ol3
				_seq_ol(root.children.last, 3, line)
				i = i + 1; next
			end
			
			# 定義リスト -------------------------------------------
			if @section == :section_dl then
				_seq_dl(root.children.last, line)
				i = i + 1; next
			end
			if @section == :section_dd then
				_seq_dd(root.children.last, line)
				i = i + 1; next
			end
			
			# ベタ書きブロック -------------------------------------
			if @section == :section_block_pre then
				if line =~ HEADMARK_BLOCK_END then
					@section = nil
				else
					HTMLScanner::Text.new(root.children.last, nil, "\n" + CGI.escapeHTML(line))
				end
				i = i + 1; next
			end

			
			# トップレベル -----------------------------------------
			# ====を使った見出し
			if ahead_line =~ HEADMARK_H2_LINE then
				_seq_h(root, 2, line)
				i = i + 2; next
			# ----を使った見出し
			elsif ahead_line =~ HEADMARK_H3_LINE then
				_seq_h(root, 3, line)
				i = i + 2; next
			else
				_seq_toplevel(root, line)
				i = i + 1; next
			end
		end
		
		
		
	#	buf = CGI.pretty(buf)
		@section = nil
		return root.to_html
	end
	
	
	
	
	
	private
	def _add_inline_element(parent_node, text)

		# HTMLEscape
		text = CGI.escapeHTML(text)


		text.gsub!(MARK_LINK_PAREN) do
			label = $1
			page_name = $2
			id = nil

			# ID指定の有無をチェック
			if page_name =~ /(.+)\#(.+)/ then
				page_name = $1
				id = $2
			elsif page_name =~ /^\#(.+)/ then
				page_name = $current_page.name
				id = $1
			end

			WikiCloth.get_page_link(label, page_name, id)
		end


		text.gsub!(MARK_LINK) do
			label = $1
			page_name = $1
			id = nil

			# ID指定の有無をチェック
			if page_name =~ /(.+)\#(.+)/ then
				page_name = $1
				id = $2
			elsif page_name =~ /^\#(.+)/ then
				page_name = $current_page.name
				id = $1
			end

			WikiCloth.get_page_link(label, page_name, id)
		end

		text.gsub!(MARK_URLLINK_PAREN) do
			url = $2
			label = $1

			# mailto: が含まれていれば、ラベルから表示を隠す
			#label = $1 if url =~ /^mailto\:(.+)/

			# @ が含まれていれば、メールアドレスとして判断
			if url =~ /\@/ then
				url = "mailto:" + url
			end

			%Q|<a href="#{url}">#{label}</a>|
		end

	
		text.gsub!(MARK_URLLINK) do
			content = $+
		  url, label = nil
			if content =~ /(.*)\|(.*)/ then
				label = $1
				url = $2
			else
				label = content
				url = content
			end

			# mailto: が含まれていれば、ラベルから表示を隠す
			#label = $1 if url =~ /^mailto\:(.+)/

			# @ が含まれていれば、メールアドレスとして判断
			if url =~ /\@/ then
				url = "mailto:" + url
			end

			%Q|<a href="#{url}">#{label}</a>|
		end
		
		text.gsub!(MARK_BR){ '<br>' }
		text.gsub!(MARK_STRONG){ '<strong>' + $+ + '</strong>' }
		text.gsub!(MARK_EM){ '<em>' + $+ + '</em>' }

		# 引用文の時だけ、citeマークアップが有効になる
		if @section == :section_blockquote then
			text.gsub!(MARK_CITE){ '<cite>' + $+ + '</cite>' }
		end

		text.gsub!(MARK_PLUGIN) do
			args = $1.split(" ")
			name = args.shift
			eval_inline_plugin(name, args)
		end

		text = transform_keywords(text)

		
		HTMLScanner::Text.new(parent_node, nil, text)


	end
	
	def _seq_toplevel(root, line)
		case line
		when HEADMARK_BLANK

		when HEADMARK_H4
			_seq_h(root, 4, $+)
		when HEADMARK_H3
			_seq_h(root, 3, $+)
		when HEADMARK_H2
			_seq_h(root, 2, $+)

		when HEADMARK_PRE
			@section = :section_pre
			pre = HTMLScanner::Element.new(root, nil, "pre")
			HTMLScanner::Text.new(pre, nil, CGI.escapeHTML($+))

		when HEADMARK_BLOCKQUOTE
			@section = :section_blockquote
			blockquote = HTMLScanner::Element.new(root, nil, "blockquote")
			_add_inline_element(blockquote, $+)
			
		when HEADMARK_UL1
			@section = :section_ul1
			ul = HTMLScanner::Element.new(root, nil, "ul")
			li = HTMLScanner::Element.new(ul, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_UL2
			@section = :section_ul2
			ul = HTMLScanner::Element.new(root, nil, "ul")
			ul = HTMLScanner::Element.new(ul, nil, "ul")
			li = HTMLScanner::Element.new(ul, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_UL3
			@section = :section_ul3
			ul = HTMLScanner::Element.new(root, nil, "ul")
			ul = HTMLScanner::Element.new(ul, nil, "ul")
			ul = HTMLScanner::Element.new(ul, nil, "ul")
			li = HTMLScanner::Element.new(ul, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_OL1
			@section = :section_ol1
			ol = HTMLScanner::Element.new(root, nil, "ol")
			li = HTMLScanner::Element.new(ol, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_OL2
			@section = :section_ol2
			ol = HTMLScanner::Element.new(root, nil, "ol")
			ol = HTMLScanner::Element.new(ol, nil, "ol")
			li = HTMLScanner::Element.new(ol, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_OL3
			@section = :section_ol3
			ol = HTMLScanner::Element.new(root, nil, "ol")
			ol = HTMLScanner::Element.new(ol, nil, "ol")
			ol = HTMLScanner::Element.new(ol, nil, "ol")
			li = HTMLScanner::Element.new(ol, nil, "li")
			_add_inline_element(li, $+)
		when HEADMARK_DL
			@section = :section_dl
			dl = HTMLScanner::Element.new(root, nil, "dl")
			dt = HTMLScanner::Element.new(dl, nil, "dt")
			_add_inline_element(dt, $+)


		when HEADMARK_PLUGIN
			args = $+.split(" ")
			name = args.shift

			HTMLScanner::Text.new(root, nil, eval_block_plugin(name, args))

		when HEADMARK_BLOCK_BEGIN
			case $+
			when 'plain', 'pre'
				@section = :section_block_pre
				HTMLScanner::Element.new(root, nil, "pre")
			when 'blockquote', 'quote'
				@section = :section_block_pre
				HTMLScanner::Element.new(root, nil, "blockquote")
			end
	
		when HEADMARK_COMMENT
			# 何もしない
			
		else
			if @section == :section_p then
				_add_inline_element(root.children.last, line)
			else
				@section = :section_p
				p = HTMLScanner::Element.new(root, nil, "p")
				_add_inline_element(p, line)
			end
		end
	end

	# 見出し共通処理
	def _seq_h(root, level, content)
		id = nil

		if content =~ MARK_ID then
			id = $1
			content.gsub!(MARK_ID, "")
		end

		h = HTMLScanner::Element.new(root, nil, "h#{level}")
		if id then
			a = HTMLScanner::Element.new(h, nil, "a", "id" => id, "name" => id)
			_add_inline_element(a, content)
		else
			_add_inline_element(h, content)
		end
		
	end
	
	# 順序無しリスト共通処理
	def _seq_ul(top_ul, current_level, line)
		li = nil

		case line
		when HEADMARK_UL1
			@section = :section_ul1
			li = HTMLScanner::Element.new(top_ul, nil, "li")
			content = $+
		when HEADMARK_UL2
			@section = :section_ul2
			if current_level >= 2 then
				ul = top_ul.children.last
			else
				ul = HTMLScanner::Element.new(top_ul, nil, "ul")
			end
			li = HTMLScanner::Element.new(ul, nil, "li")
			content = $+
		when HEADMARK_UL3
			@section = :section_ul3
			case current_level
			when 1
				ul = HTMLScanner::Element.new(top_ul, nil, "ul")
				ul = HTMLScanner::Element.new(ul, nil, "ul")
			when 2
				ul = top_ul.children.last
				ul = HTMLScanner::Element.new(ul, nil, "ul")
			when 3
				ul = top_ul.children.last.children.last
			end
			li = HTMLScanner::Element.new(ul, nil, "li")
			content = $+
		when HEADMARK_BLANK
			@section = nil
			return
			
		else
			case current_level
			when 1
				li = top_ul.children.last
			when 2
				li = top_ul.children.last.children.last
			when 3
				li = top_ul.children.last.children.last.children.last
			end
			content = line.gsub(/^[ ]*/, "")
		end

		_add_inline_element(li, content)
	end
	
	
	def _seq_ol(top_ol, current_level, line)
		li = nil

		case line
		when HEADMARK_OL1
			@section = :section_ol1
			li = HTMLScanner::Element.new(top_ol, nil, "li")
			content = $+

		when HEADMARK_OL2
			@section = :section_ol2
			if current_level >= 2 then
				ol = top_ol.children.last
			else
				ol = HTMLScanner::Element.new(top_ol, nil, "ol")
			end
			li = HTMLScanner::Element.new(ol, nil, "li")
			content = $+

		when HEADMARK_OL3
			@section = :section_ol3
			case current_level
			when 1
				ol = HTMLScanner::Element.new(top_ol, nil, "ol")
				ol = HTMLScanner::Element.new(ol, nil, "ol")
			when 2
				ol = top_ol.children.last
				ol = HTMLScanner::Element.new(ol, nil, "ol")
			when 3
				ol = top_ol.children.last.children.last
			end
			li = HTMLScanner::Element.new(ol, nil, "li")
			content = $+

		when HEADMARK_BLANK
			@section = nil
			return

		else
			case current_level
			when 1
				li = top_ol.children.last
			when 2
				li = top_ol.children.last.children.last
			when 3
				li = top_ol.children.last.children.last.children.last
			end
			
			content = line.gsub(/^[ ]*/, "")
		end

		_add_inline_element(li, content)
	end

	def _seq_dl(dl, line)
		case line
		when HEADMARK_DL
			dt = HTMLScanner::Element.new(dl, nil, "dt")
			_add_inline_element(dt, $+)
		when HEADMARK_BLANK
			@section = nil
			return
		else
			@section = :section_dd
			dd = HTMLScanner::Element.new(dl, nil, "dd")
			_add_inline_element(dd, line)
		end
	end
	
	def _seq_dd(dl, line)
		case line
		when HEADMARK_DL
			@section = :section_dl
			dt = HTMLScanner::Element.new(dl, nil, "dt")
			_add_inline_element(dt, $+)
		when HEADMARK_BLANK
			@section = nil
			return
		else
			_add_inline_element(dl.children.last, line)
		end
	end

	public
	def redirecting_page_name
		return $1 if @text =~ /^\{\{redirect (.*)\}\}[ ]*$/
	end

	def remove_plugin(plugin_name)
		return WikiCloth.new(@text.gsub(/\{\{#{Regexp.escape(plugin_name)}\}\}/, ""))
	end

	COMMENT_PLUGIN_PATTERN = /^\{\{[ ]*comment[ ]*\}\}$/
	def insert_comment(index, comment_text, commenter)
		i = 0

		text = @text.dup
		text.gsub!(COMMENT_PLUGIN_PATTERN) do
			if i == index then
				# コメントを挿入
				i += 1
				sprintf("%s\n* %s - %s (%s)", $~[0], comment_text, commenter, RevWiki.time_to_str(Time.now))
			else
				# スキップ
				i += 1
				$~[0]
			end
		end

		return WikiCloth.new(text)
	end

end


def RevWiki.test; print(WikiCloth.new(<<ECHO).content_to_html)
原則
====================
* 半角スペースは全て捨てられます。
  ただし、「ベタ書き」のときだけは有効です（後述）。
* 空行が文書の
* 文章の途中で改行しても
* 行の先頭に記号を置くことによって、さまざまなものを表現できます。

見出しを書く
====================
また、見出しのタイトルを先に書いて、次の行に二つ以上の = や - を重ねると、これも見出しになります。

|見出しレベル１
|==============
|
|見出しレベル２
|--------------

リストを作る
====================
行頭に*を置くと番号無しリスト、+を置くと番号付きリストになります。

==plain
* りんご
* みかん
* すいか

+ リスト中では、文書の
  途中で改行してもよい。
+ 記号と文書の間に、半角スペースを挟んでもよい。
  インデントも自由。
+ 空行までが一つのリストになる。
==

* りんご
* みかん
* すいか

+ リスト中では、文書の
  途中で改行してもよい。
+ 記号と文書の間に、半角スペースを挟んでもよい。
  インデントも自由。
+ 空行までが一つのリストになる。

また、番号付きリストに限り、次のような書き方も可能です。

|1.docomo
|2.au
|3.vodafone
|4.tu-ka

1.docomo
2.au
3.vodafone
4.tu-ka
ECHO
end
